;CreateView :            ViewQuat xyzw <== ViewVec xyz0
;CreateMatrix :         takes in an array of 16 floats to fill in a 4x4 homogeneous matrix from a quaternion
;CreateFromMatrix : takes a 3x3 or 4x4 matrix and converts it to a quaternion, depending on rowColumnCount
;CreateRotationAroundAxis ;Creates a Rotation quat from an angle and vec3-based axis
;Slerp:                      returns a spherical linear interpolated quaternion between q1 and q2, according to ft
;Normalize               makes a quat unit length (important for rotations)

class CQuaternion, ,C++ compatible
    void Conjugate:pSourceQuat
    void CreateView                         ;<-- Create View Quaternion from View vector
    void CreateMatrix:pMatrix         ;<-- Convert Quat to Matrix
    void CreateFromMatrix:pMatrix, rowColumnCount   ;<-- Convert Matrix(3 or 4) to Quat
    void CreateRotationAroundAxis:fAngle, pvAxis ;<-- Create Rotation Quat for arb axis
    void Slerp:pQ1, pQ2, ft
    void Normalize
    void Multiply:pA,pB
    float fX                         ; This stores the 4D values for the quaternion
    float fY
    float fZ
    float fW
endclass

;This proc creates a Conjugate from another quat
CQuaternion_Conjugate proc uses esi pQuat
mov esi, pQuat
fld [esi].CQuaternion.fX
fchs
fstp [ecx].CQuaternion.fX
fld [esi].CQuaternion.fY
fchs
fstp [ecx].CQuaternion.fY
fld [esi].CQuaternion.fZ
fchs
fstp [ecx].CQuaternion.fZ
ret
CQuaternion_Conjugate endp


CQuaternion_CreateView proc uses esi pViewVector
mov esi,pViewVector
fld [esi].Vec3.X
fstp [ecx].CQuaternion.fX
fld [esi].Vec3.Y
fstp [ecx].CQuaternion.fY
fld [esi].Vec3.Z
fstp [ecx].CQuaternion.fZ
fld r4_0_0
fstp [ecx].CQuaternion.fW
ret
CQuaternion_CreateView endp

CQuaternion_CreateRotationAroundAxis proc uses esi fAngle, pvAxis
local thetadiv2:REAL8

fld fAngle
fmul r4_0_5
fstp  thetadiv2

;R.x = A.x * sin(theta/2)
;R.y = A.y * sin(theta/2)
;R.z = A.z * sin(theta/2)
;R.w = cos(theta/2)

mov esi, pvAxis
assume esi:ptr Vec3
fld thetadiv2
fsin
fmul [esi].X
fstp [ecx].CQuaternion.fX

fld thetadiv2
fsin
fmul [esi].Y
fstp [ecx].CQuaternion.fY

fld thetadiv2
fsin
fmul [esi].Z
fstp [ecx].CQuaternion.fZ

fld thetadiv2
fcos
fstp [ecx].CQuaternion.fW

assume esi:nothing
ret
CQuaternion_CreateRotationAroundAxis endp



;============================================================
;This method multiples two OTHER quaternions (given as pA and pB)
;into THIS target quaternion
CQuaternion_Multiply proc uses esi edi pA, pB
mov esi,pA
mov edi,pB
assume esi:ptr CQuaternion
assume edi:ptr CQuaternion
assume ecx:ptr CQuaternion

;C.x = | A.w*B.x + A.x*B.w + A.y*B.z - A.z*B.y |
fld [esi].fW
fmul [edi].fX
fld [esi].fX
fmul [edi].fW
fadd
fld [esi].fY
fmul [edi].fZ
fadd
fld [esi].fZ
fmul [edi].fY
fsub
fstp [ecx].fX

;C.y = | A.w*B.y - A.x*B.z + A.y*B.w + A.z*B.x |
fld [esi].fW
fmul [edi].fY
fld [esi].fX
fmul [edi].fZ
fsub
fld [esi].fY
fmul [edi].fW
fadd
fld [esi].fZ
fmul [edi].fX
fadd
fstp [ecx].fY

;C.z = | A.w*B.z + A.x*B.y - A.y*B.x + A.z*B.w |
fld [esi].fW
fmul [edi].fZ
fld [esi].fX
fmul [edi].fY
fadd
fld [esi].fY
fmul [edi].fX
fsub
fld [esi].fZ
fmul [edi].fW
fadd
fstp [ecx].fZ

;C.w = | A.w*B.w - A.x*B.x - A.y*B.y - A.z*B.z |
fld [esi].fW
fmul [edi].fW
fld [esi].fX
fmul [edi].fX
fsub
fld [esi].fY
fmul [edi].fY
fsub
fld [esi].fZ
fmul [edi].fZ
fsub
fstp  [ecx].fW

assume esi:nothing
assume edi:nothing
assume ecx:nothing
ret
CQuaternion_Multiply endp
;======================================
;This procedure Normalizes a quaternion,
;which gives us a quaternion of length 1 (which is very important for rotations). 
;======================================
CQuaternion_Normalize proc
local fQ:REAL8 
;First calculate Q length of Quat, where Q=sqrt(w^2+x^2+y^2+z^)
fld [ecx].CQuaternion.fW
fmul [ecx].CQuaternion.fW
fld [ecx].CQuaternion.fX
fmul [ecx].CQuaternion.fX
fadd
fld [ecx].CQuaternion.fY
fmul [ecx].CQuaternion.fY
fadd
fld [ecx].CQuaternion.fZ
fmul [ecx].CQuaternion.fZ
fadd
fsqrt
fstp fQ
;Normalize Quat by dividing its components by Q
fld [ecx].CQuaternion.fW
fdiv fQ
fstp [ecx].CQuaternion.fW
fld [ecx].CQuaternion.fX
fdiv fQ
fstp [ecx].CQuaternion.fX
fld [ecx].CQuaternion.fY
fdiv fQ
fstp [ecx].CQuaternion.fY
fld [ecx].CQuaternion.fZ
fdiv fQ
fstp [ecx].CQuaternion.fZ
ret
CQuaternion_Normalize endp

;======================================
; This is our default constructor, which initializes everything to an identity
; quaternion.  An identity quaternion has x, y, z as 0 and w as 1.
CQuaternion_CQuaternion proc 
    mov [ecx].CQuaternion.fX,0
    mov [ecx].CQuaternion.fY,0
    mov [ecx].CQuaternion.fZ,0
    fld1
    fstp [ecx].CQuaternion.fW
    ret
CQuaternion_CQuaternion endp
CQuaternion_$CQuaternion proc
    ret
CQuaternion_$CQuaternion endp

;===========================================================================
;This function converts a quaternion to a rotation matrix
CQuaternion_CreateMatrix proc uses esi pMatrix
; Make sure the matrix has allocated memory to store the rotation data
    .if !pMatrix
         ret
    .endif
    assume ecx:ptr CQuaternion
    mov esi,pMatrix
; Fill in the rows of the 4x4 matrix, according to the quaternion to matrix equations
; First row
;pMatrix[ 0] = 1.0f - 2.0f * ( y * y + z * z );  
    fld1                ;load st(0) with 1.0f (push 1.0f onto fpu stack)
    fld r4_2_0      ;load st(0) with 2.0f , now st(0)=2.0f st(1)=1.0f
    fld [ecx].CQuaternion.fY     ;st(0) = y , st(1) = 2.0f , st(2) = 1.0f
    fmul [ecx].CQuaternion.fY  ;st(0) = y*y , st(1) = 2.0f , st(2) = 1.0f
    fld [ecx].CQuaternion.fZ     ;st(0) = z , st(1) = y*y , st(2) = 2.0f , st(3) = 1.0f
    fmul [ecx].CQuaternion.fZ  ;st(0) = z*z , st(1) = y*y , st(2) = 2.0f , st(3) = 1.0f
    fadd               ;st(0) = (y*y+z*z) ,  st(1) = 2.0f , st(2) = 1.0f
    fmul               ;st(0) = 2.0f * (y*y+z*z) ,  st(1) = 1.0f
    fsub               ;st(0) = 1.0 - 2.0f * (y*y+z*z) the only thing left on the fpu stack now is the answer
    fstp REAL4 ptr [esi+(0*4)]    ;Now store the answer, yay !!

;pMatrix[ 1] = 2.0f * ( x * y - w * z );  
    fld [ecx].CQuaternion.fX
    fmul [ecx].CQuaternion.fY
    fld [ecx].CQuaternion.fW
    fmul [ecx].CQuaternion.fZ
    fsub
    fmul r4_2_0
    fstp REAL4 ptr [esi+(1*4)]

;pMatrix[ 2] = 2.0f * ( x * z + w * y );  
    fld [ecx].CQuaternion.fX
    fmul [ecx].CQuaternion.fZ
    fld [ecx].CQuaternion.fW
    fmul [ecx].CQuaternion.fY
    fadd
    fmul r4_2_0
    fstp REAL4 ptr [esi+(2*4)]

;pMatrix[ 3] = 0.0f;  
    mov REAL4 ptr [esi+(3*4)],0

; Second row
;pMatrix[ 4] = 2.0f * ( x * y + w * z );  
    fld [ecx].CQuaternion.fX
    fmul [ecx].CQuaternion.fY
    fld [ecx].CQuaternion.fW
    fmul [ecx].CQuaternion.fZ
    fadd
    fmul r4_2_0
    fstp REAL4 ptr [esi+(4*4)]

;pMatrix[ 5] = 1.0f - 2.0f * ( x * x + z * z );  
    fld1
    fld r4_2_0
    fld [ecx].CQuaternion.fX
    fmul [ecx].CQuaternion.fX 
    fld [ecx].CQuaternion.fZ 
    fmul [ecx].CQuaternion.fZ 
    fadd  
    fmul
    fsub 
    fstp REAL4 ptr [esi+(5*4)] 

;pMatrix[ 6] = 2.0f * ( y * z - w * x );  
    fld [ecx].CQuaternion.fY
    fmul [ecx].CQuaternion.fZ
    fld [ecx].CQuaternion.fW
    fmul [ecx].CQuaternion.fZ
    fsub
    fmul r4_2_0
    fstp REAL4 ptr [esi+(6*4)]

;pMatrix[ 7] = 0.0f;
    mov REAL4 ptr [esi+(7*4)],0

; Third row
;pMatrix[ 8] = 2.0f * ( x * z - w * y );  
    fld [ecx].CQuaternion.fX
    fmul [ecx].CQuaternion.fZ
    fld [ecx].CQuaternion.fW
    fmul [ecx].CQuaternion.fY
    fsub
    fmul r4_2_0
    fstp REAL4 ptr [esi+(8*4)]

;pMatrix[ 9] = 2.0f * ( y * z + w * x );  
    fld [ecx].CQuaternion.fY
    fmul [ecx].CQuaternion.fZ
    fld [ecx].CQuaternion.fW
    fmul [ecx].CQuaternion.fX
    fadd
    fmul r4_2_0
    fstp REAL4 ptr [esi+(9*4)]

;pMatrix[10] = 1.0f - 2.0f * ( x * x + y * y );  
    fld1
    fld r4_2_0
    fld [ecx].CQuaternion.fX
    fmul [ecx].CQuaternion.fX 
    fld [ecx].CQuaternion.fY
    fmul [ecx].CQuaternion.fY
    fadd  
    fmul
    fsub 
    fstp REAL4 ptr [esi+(10*4)] 

;pMatrix[11] = 0.0f;  
    xor eax,eax
    mov REAL4 ptr [esi+(11*4)],eax

; Fourth row
;pMatrix[12] = 0;  
    mov REAL4 ptr [esi+(12*4)],eax
;pMatrix[13] = 0;  
    mov REAL4 ptr [esi+(13*4)],eax
;pMatrix[14] = 0;  
    mov REAL4 ptr [esi+(14*4)],eax
;pMatrix[15] = 1.0f;
    fld1
    fstp REAL4 ptr [esi+(15*4)]

; Now pMatrix[] is a 4x4 homogeneous matrix that can be applied to an OpenGL Matrix
    assume ecx:nothing
    ret
CQuaternion_CreateMatrix endp
;===========================================================================

;This creates a quaternion from a 3x3 or a 4x4 matrix, depending on rowColumnCount
CQuaternion_CreateFromMatrix proc pTheMatrix, rowColumnCount
local fdiagonal:REAL4
local fTheScale:REAL4
local pMatrix
local m4x4[16]:REAL4        ; Create a 4x4 matrix to convert a 3x3 matrix to a 4x4 matrix (If rowColumnCount == 3)

; Make sure the matrix has valid memory and it's not expected that we allocate it.
; Also, we do a check to make sure the matrix is a 3x3 or a 4x4 (must be 3 or 4).
    .if (!pTheMatrix || ((rowColumnCount != 3) && (rowColumnCount != 4))) 
        ret
    .endif

; This function is used to take in a 3x3 or 4x4 matrix and convert the matrix
; to a quaternion.  If rowColumnCount is a 3, then we need to convert the 3x3
; matrix passed in to a 4x4 matrix, otherwise we just leave the matrix how it is.
; Since we want to apply a matrix to an OpenGL matrix, we need it to be 4x4.

; Point the matrix pointer to the matrix passed in, assuming it's a 4x4 matrix
    m2m pMatrix , pTheMatrix

; If the matrix is a 3x3 matrix (which it is for Quake3), then convert it to a 4x4
    .if rowColumnCount == 3
        ; Set the 9 top left indices of the 4x4 matrix to the 9 indices in the 3x3 matrix.
        ; It would be a good idea to actually draw this out so you can visualize it.
        ;m4x4[0]  = pTheMatrix[0];	m4x4[1]  = pTheMatrix[1];	m4x4[2]  = pTheMatrix[2];
        mov esi,pTheMatrix
        lea edi, m4x4
        m2m REAL4 ptr [edi+(0*4)], REAL4 ptr [esi+(0*4)]
        m2m REAL4 ptr [edi+(1*4)], REAL4 ptr [esi+(1*4)]
        m2m REAL4 ptr [edi+(2*4)], REAL4 ptr [esi+(2*4)]
        mov REAL4 ptr [edi+(3*4)],0
        ;m4x4[4]  = pTheMatrix[3];	m4x4[5]  = pTheMatrix[4];	m4x4[6]  = pTheMatrix[5];
        m2m REAL4 ptr [edi+(4*4)], REAL4 ptr [esi+(3*4)]
        m2m REAL4 ptr [edi+(5*4)], REAL4 ptr [esi+(4*4)]
        m2m REAL4 ptr [edi+(6*4)], REAL4 ptr [esi+(5*4)]
        mov REAL4 ptr [edi+(7*4)],0
        ;m4x4[8]  = pTheMatrix[6];	m4x4[9]  = pTheMatrix[7];	m4x4[10] = pTheMatrix[8];
        m2m REAL4 ptr [edi+(8*4)], REAL4 ptr [esi+(6*4)]
        m2m REAL4 ptr [edi+(9*4)], REAL4 ptr [esi+(7*4)]
        m2m REAL4 ptr [edi+(10*4)], REAL4 ptr [esi+(8*4)]
        mov REAL4 ptr [edi+(11*4)],0
; Since the bottom and far right indices are zero, set the bottom right corner to 1.
; This is so that it follows the standard diagonal line of 1's in the identity matrix.
        ;m4x4[15] = 1;
        xor eax,eax
        mov REAL4 ptr [edi+(12*4)],eax
        mov REAL4 ptr [edi+(13*4)],eax
        mov REAL4 ptr [edi+(14*4)],eax
        fld1
        fstp  REAL4 ptr [edi+(15*4)]
        ; Set the matrix pointer to the first index in the newly converted matrix
        mov pMatrix, edi
    .endif
; The next step, once we made sure we are dealing with a 4x4 matrix, is to check the
; diagonal of the matrix.  This means that we add up all of the indices that comprise
; the standard 1's in the identity matrix.  If you draw out the identity matrix of a
; 4x4 matrix, you will see that they 1's form a diagonal line.  Notice we just assume
; that the last index (15) is 1 because it is not effected in the 3x3 rotation matrix.

; Find the diagonal of the matrix by adding up it's diagonal indices.
; This is also known as the "trace", but I will call the variable diagonal.
;float diagonal = pMatrix[0] + pMatrix[5] + pMatrix[10] + 1;
    fld1
    fadd REAL4 ptr [edi]
    fadd REAL4 ptr [edi+(5*4)]
    fadd REAL4 ptr [edi+(10*4)]
    fstp fdiagonal      
;float scale = 0.0f;
    mov fTheScale, 0

; Below we check if the diagonal is greater than zero.  To avoid accidents with
; floating point numbers, we substitute 0 with 0.00000001.  If the diagonal is
; great than zero, we can perform an "instant" calculation, otherwise we will need
; to identify which diagonal element has the greatest value.  Note, that it appears
; that %99 of the time, the diagonal IS greater than 0 so the rest is rarely used.

; If the diagonal is greater than zero 
    mov eax,fdiagonal
    .if  (eax > 0) 
    and eax,80000000h
    .if (eax== 0)   ;if its not 0 and its not negative :)
        ; Calculate the scale of the diagonal
        fld fdiagonal
        fsqrt
        fmul r4_2_0
        fstp fTheScale ;        fTheScale = float(sqrt(diagonal ) * 2);

; Calculate the x, y, x and w of the quaternion through the respective equation
       ; x = ( pMatrix[9] - pMatrix[6] ) / scale;
        fld REAL4 ptr [edi+(9*4)]
        fsub REAL4 ptr [edi+(6*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fX
        ;y = ( pMatrix[2] - pMatrix[8] ) / scale;
        fld REAL4 ptr [edi+(2*4)]
        fsub REAL4 ptr [edi+(8*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fY
        ;z = ( pMatrix[4] - pMatrix[1] ) / scale;
        fld REAL4 ptr [edi+(4*4)]
        fsub REAL4 ptr [edi+(1*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fZ
        ;w = 0.25f * scale;
        fld r4_0_25
        fmul fTheScale
        fstp [ecx].CQuaternion.fW
        jmp PastThisShit
    .endif
    .endif

        ; If the first element of the diagonal is the greatest value
        ;if ( pMatrix[0] > pMatrix[5] && pMatrix[0] > pMatrix[10] )  
        fld REAL4 ptr [edi]
        fcomp REAL4 ptr [edi+(5*4)]
        __FJLE @F
        fld REAL4 ptr [edi]
        fcomp REAL4 ptr [edi+(10*4)]
        __FJLE @F
; Find the scale according to the first element, and double that value
;scale  = (float)sqrt( 1.0f + pMatrix[0] - pMatrix[5] - pMatrix[10] ) * 2.0f;
        fld1
        fadd REAL4 ptr [edi]
        fsub REAL4 ptr [edi+(5*4)]
        fsub REAL4 ptr [edi+(10*4)]
        fsqrt
        fmul r4_2_0
        fstp fTheScale
; Calculate the x, y, x and w of the quaternion through the respective equation
        ;x = 0.25f * scale;
        fld r4_0_25
        fmul fTheScale
        fstp [ecx].CQuaternion.fX
        ;y = (pMatrix[4] + pMatrix[1] ) / scale;
        fld REAL4 ptr [edi+(4*4)]
        fadd REAL4 ptr [edi+(1*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fY
        ;z = (pMatrix[2] + pMatrix[8] ) / scale;
        fld REAL4 ptr [edi+(2*4)]
        fadd REAL4 ptr [edi+(8*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fZ
        ;w = (pMatrix[9] - pMatrix[6] ) / scale;	 
        fld REAL4 ptr [edi+(9*4)]
        fsub REAL4 ptr [edi+(6*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fW
        jmp PastThisShit
@@:
        ; Else if the second element of the diagonal is the greatest value
        ;else if ( pMatrix[5] > pMatrix[10] ) 
        fld REAL4 ptr [edi+(5*4)]
        fcomp REAL4 ptr [edi+(10*4)]
        __FJLE @F
; Find the scale according to the second element, and double that value
;scale  = (float)sqrt( 1.0f + pMatrix[5] - pMatrix[0] - pMatrix[10] ) * 2.0f;
        fld1
        fadd REAL4 ptr [edi+(5*4)]
        fsub REAL4 ptr [edi+(0*4)]
        fsub REAL4 ptr [edi+(10*4)]
        fsqrt
        fmul r4_2_0
        fstp fTheScale
; Calculate the x, y, x and w of the quaternion through the respective equation
        ;x = (pMatrix[4] + pMatrix[1] ) / scale
        fld REAL4 ptr [edi+(4*4)]
        fadd REAL4 ptr [edi+(1*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fX
        ;y = 0.25f * scale;
        fld r4_0_25
        fmul fTheScale
        fstp [ecx].CQuaternion.fY
        ;z = (pMatrix[9] + pMatrix[6] ) / scale;
        fld REAL4 ptr [edi+(9*4)]
        fadd REAL4 ptr [edi+(6*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fZ
        ;w = (pMatrix[2] - pMatrix[8] ) / scale;
        fld REAL4 ptr [edi+(2*4)]
        fsub REAL4 ptr [edi+(8*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fW
        jmp PastThisShit
@@:
        ; Else the third element of the diagonal is the greatest value
        ;else 
; Find the scale according to the third element, and double that value
;scale  = (float)sqrt( 1.0f + pMatrix[10] - pMatrix[0] - pMatrix[5] ) * 2.0f;
        fld1
        fadd REAL4 ptr [edi+(10*4)]
        fsub REAL4 ptr [edi+(0*4)]
        fsub REAL4 ptr [edi+(5*4)]
        fsqrt
        fmul r4_2_0
        fstp fTheScale
; Calculate the x, y, x and w of the quaternion through the respective equation
        ;x = (pMatrix[2] + pMatrix[8] ) / scale;
        fld REAL4 ptr [edi+(2*4)]
        fadd REAL4 ptr [edi+(8*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fX
        ;y = (pMatrix[9] + pMatrix[6] ) / scale;
        fld REAL4 ptr [edi+(9*4)]
        fadd REAL4 ptr [edi+(6*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fY
        ;z = 0.25f * scale;
        fld r4_0_25
        fmul fTheScale
        fstp [ecx].CQuaternion.fZ
        ;w = (pMatrix[4] - pMatrix[1] ) / scale;
        fld REAL4 ptr [edi+(4*4)]
        fsub REAL4 ptr [edi+(1*4)]
        fdiv fTheScale
        fstp [ecx].CQuaternion.fY
PastThisShit:
    assume ecx:nothing
    ret
CQuaternion_CreateFromMatrix endp
;===========================================================================
;Returns a spherical linear interpolated quaternion between q1 and q2, with respect to ft
CQuaternion_Slerp proc pQ1, pQ2, ft
; Create a local quaternion to store the interpolated quaternion
local qInterpolated , me, fresult, fTheScale0, fTheScale1, theta, sinTheta

    mov me,ecx

; This function is the milk and honey of our quaternion code, the rest of
; the functions are an appendage to what is done here.  Everyone understands
; the terms, "matrix to quaternion", "quaternion to matrix", "create quaternion matrix",
; "quaternion multiplication", etc.. but "SLERP" is the stumbling block, even a little 
; bit after hearing what it stands for, "Spherical Linear Interpolation".  What that
; means is that we have 2 quaternions (or rotations) and we want to interpolate between 
; them.  The reason what it's called "spherical" is that quaternions deal with a sphere.  
; Linear interpolation just deals with 2 points primarily, where when dealing with angles
; and rotations, we need to use sin() and cos() for interpolation.  If we wanted to use
; quaternions for camera rotations, which have much more instant and jerky changes in 
; rotations, we would use Spherical-Cubic Interpolation.  The equation for SLERP is this:
;
; q = (((b.a)^-1)^t)a
;
; Go here for an a detailed explanation and proofs of this equation:
;
; http:;www.magic-software.com/Documentation/quat.pdf
;
; Now, Let's code it

; Here we do a check to make sure the 2 quaternions aren't the same, return q1 if they are
    mov esi, pQ1
    mov edi,pQ2
    mov eax,[esi].CQuaternion.fX
    .if eax == [edi].CQuaternion.fX
        mov eax, [esi].CQuaternion.fY
        .if eax == [edi].CQuaternion.fY
            mov eax, [esi].CQuaternion.fZ
            .if eax == [edi].CQuaternion.fZ
                mov eax,[esi].CQuaternion.fW
                .if eax == [edi].CQuaternion.fW                        
		return pQ1
                .endif
            .endif
        .endif
    .endif

    mov qInterpolated, new (CQuaternion)    ;Create a return instance
; Following the (b.a) part of the equation, we do a dot product between q1 and q2.
; We can do a dot product because the same math applied for a 3D vector as a 4D vector.
;float result = (q1.x * q2.x) + (q1.y * q2.y) + (q1.z * q2.z) + (q1.w * q2.w);
    fld [esi].CQuaternion.fX
    fmul [edi].CQuaternion.fX
    fld [esi].CQuaternion.fY
    fmul [edi].CQuaternion.fY
    fld [esi].CQuaternion.fZ
    fmul [edi].CQuaternion.fZ
    fld [esi].CQuaternion.fW
    fmul [edi].CQuaternion.fW
    fadd
    fadd
    fadd
    fstp fresult
; If the dot product is less than 0, the angle is greater than 90 degrees
    mov eax,fresult
    and eax,80000000h
    .if eax != 0    ;check bit 31 for negative value
        ; Negate the second quaternion and the result of the dot product
        ;q2 = CQuaternion(-q2.x, -q2.y, -q2.z, -q2.w);
        fld  [edi].CQuaternion.fX
        fchs
        fstp [edi].CQuaternion.fX
        fld  [edi].CQuaternion.fY
        fchs
        fstp [edi].CQuaternion.fY
        fld  [edi].CQuaternion.fZ
        fchs
        fstp [edi].CQuaternion.fZ
        fld  [edi].CQuaternion.fW
        fchs
        fstp [edi].CQuaternion.fW
        ;result = -result;
        fld fresult
        fchs
        fstp fresult
    .endif

; Set the first and second scale for the interpolation
    ;float scale0 = 1 - t, scale1 = t;
    fld1
    fsub ft
    fstp fTheScale0
    m2m fTheScale1, ft

; Next, we want to actually calculate the spherical interpolation.  Since this
; calculation is quite computationally expensive, we want to only perform it
; if the angle between the 2 quaternions is large enough to warrant it.  If the
; angle is fairly small, we can actually just do a simpler linear interpolation
; of the 2 quaternions, and skip all the complex math.  We create a "delta" value
; of 0.1 to say that if the cosine of the angle (result of the dot product) between
; the 2 quaternions is smaller than 0.1, then we do NOT want to perform the full on 
; interpolation using.  This is because you won't really notice the difference.

; Check if the angle between the 2 quaternions was big enough to warrant such calculations
    fld1
    fsub fresult
    fcomp r4_0_1    ; if (1 - result > 0.1f)
    __FJLE @F
; Get the angle between the 2 quaternions, and then store the sin() of that angle
    ;float theta = (float)acos(result);
    facos (fresult)
    fstp theta
    ;float sinTheta = (float)sin(theta);
    fld fresult
    fsin
    fstp sinTheta
; Calculate the scale for q1 and q2, according to the angle and it's sine value
    ;scale0 = (float)sin( ( 1 - t ) * theta) / sinTheta;
    fld1
    fsub ft
    fmul theta
    fsin
    fdiv sinTheta
    fstp fTheScale0

    ;scale1 = (float)sin( ( t * theta) ) / sinTheta;
    fld ft
    fmul theta
    fsin
    fdiv sinTheta
    fstp fTheScale1

@@:
PastThisShit:
; Calculate the x, y, z and w values for the quaternion by using a special
; form of linear interpolation for quaternions.
    mov ebx,qInterpolated
    ;qInterpolated.x = (scale0 * q1.x) + (scale1 * q2.x);
    fld fTheScale0
    fmul [esi].CQuaternion.fX
    fld fTheScale1
    fmul [edi].CQuaternion.fX
    fadd
    fstp [ebx].CQuaternion.fX
    ;qInterpolated.y = (scale0 * q1.y) + (scale1 * q2.y);
    fld fTheScale0
    fmul [esi].CQuaternion.fY
    fld fTheScale1
    fmul [edi].CQuaternion.fY
    fadd
    fstp [ebx].CQuaternion.fY
    ;qInterpolated.z = (scale0 * q1.z) + (scale1 * q2.z);
    fld fTheScale0
    fmul [esi].CQuaternion.fZ
    fld fTheScale1
    fmul [edi].CQuaternion.fZ
    fadd
    fstp [ebx].CQuaternion.fZ
    ;qInterpolated.w = (scale0 * q1.w) + (scale1 * q2.w);
    fld fTheScale0
    fmul [esi].CQuaternion.fW
    fld fTheScale1
    fmul [edi].CQuaternion.fW
    fadd
    fstp [ebx].CQuaternion.fW
; Return the interpolated quaternion
    return qInterpolated
CQuaternion_Slerp endp
